<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . '/../config/db.php';

$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        $recipe_id = $_GET['recipe_id'] ?? null;
        $slug = $_GET['slug'] ?? null;
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $limit = 10;
        $offset = ($page - 1) * $limit;

        if ($slug && !$recipe_id) {
            // Get recipe ID from slug
            $stmt = $pdo->prepare("SELECT id FROM recipes WHERE slug = :slug");
            $stmt->execute(['slug' => $slug]);
            $recipe = $stmt->fetch();
            if ($recipe) {
                $recipe_id = $recipe['id'];
            } else {
                echo json_encode(["comments" => [], "has_more" => false]);
                exit;
            }
        }

        if ($recipe_id) {
            // 1. Count Total Root Comments (Approved)
            $countStmt = $pdo->prepare("SELECT COUNT(*) FROM comments WHERE recipe_id = :recipe_id AND parent_id IS NULL AND is_approved = TRUE");
            $countStmt->execute(['recipe_id' => $recipe_id]);
            $totalRoots = $countStmt->fetchColumn();

            // 2. Fetch Root Comments (Approved, Paginated)
            // Note: PDO LIMIT with parameters can be tricky in emulation mode, so checking int casting
            $stmt = $pdo->prepare("SELECT * FROM comments WHERE recipe_id = :recipe_id AND parent_id IS NULL AND is_approved = TRUE ORDER BY created_at DESC LIMIT :limit OFFSET :offset");
            $stmt->bindValue(':recipe_id', $recipe_id, PDO::PARAM_INT);
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            $roots = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // 3. Fetch Replies for each Root
            foreach ($roots as &$root) {
                $rStmt = $pdo->prepare("SELECT * FROM comments WHERE parent_id = :parent_id AND is_approved = TRUE ORDER BY created_at ASC");
                $rStmt->execute(['parent_id' => $root['id']]);
                $root['replies'] = $rStmt->fetchAll(PDO::FETCH_ASSOC);
            }

            echo json_encode([
                "comments" => $roots,
                "has_more" => ($totalRoots > ($offset + $limit)),
                "total" => $totalRoots,
                "page" => $page
            ]);
        } else {
            http_response_code(400);
            echo json_encode(["error" => "Recipe ID or Slug is required"]);
        }
        break;

    case 'POST':
        $data = json_decode(file_get_contents("php://input"));
        
        // Supports adding by slug if ID is not provided
        if (!empty($data->slug) && empty($data->recipe_id)) {
            $stmt = $pdo->prepare("SELECT id FROM recipes WHERE slug = :slug");
            $stmt->execute(['slug' => $data->slug]);
            $recipe = $stmt->fetch();
            if ($recipe) {
                $data->recipe_id = $recipe['id'];
            }
        }

        if (!empty($data->recipe_id) && !empty($data->user_name) && !empty($data->comment_text)) {
            try {
                $stmt = $pdo->prepare("INSERT INTO comments (recipe_id, user_name, comment_text, parent_id, is_approved) VALUES (:recipe_id, :user_name, :comment_text, :parent_id, :is_approved)");
                $stmt->execute([
                    ':recipe_id' => $data->recipe_id,
                    ':user_name' => htmlspecialchars(strip_tags($data->user_name)),
                    ':comment_text' => htmlspecialchars(strip_tags($data->comment_text)),
                    ':parent_id' => !empty($data->parent_id) ? $data->parent_id : null,
                    ':is_approved' => 0 // Force pending approval
                ]);
                echo json_encode(["status" => "success", "message" => "Comment added", "id" => $pdo->lastInsertId()]);
            } catch (PDOException $e) {
                http_response_code(500);
                echo json_encode(["error" => "Database error: " . $e->getMessage()]);
            }
        } else {
            http_response_code(400);
            echo json_encode(["error" => "Missing required fields"]);
        }
        break;
}
?>
