<?php
// Secure output buffering to prevent accidental output
ob_start();

// Error handling for JSON output
error_reporting(E_ALL);
ini_set('display_errors', 0);

set_error_handler(function($severity, $message, $file, $line) {
    if (error_reporting() & $severity) {
        throw new ErrorException($message, 0, $severity, $file, $line);
    }
});

set_exception_handler(function($e) {
    if (ob_get_length()) ob_end_clean();
    http_response_code(500);
    header("Content-Type: application/json; charset=UTF-8");
    echo json_encode([
        "error" => "Internal Server Error",
        "message" => $e->getMessage(),
        "file" => basename($e->getFile()),
        "line" => $e->getLine()
    ]);
    exit;
});

// Register shutdown function to catch fatal errors
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== NULL && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        if (ob_get_length()) ob_end_clean();
        http_response_code(500);
        header("Content-Type: application/json; charset=UTF-8");
        echo json_encode([
            "error" => "Fatal Error",
            "message" => $error['message'],
            "file" => basename($error['file']),
            "line" => $error['line']
        ]);
    }
});

// Early CORS headers
if (isset($_SERVER['HTTP_ORIGIN'])) {
    header("Access-Control-Allow-Origin: {$_SERVER['HTTP_ORIGIN']}");
} else {
    header("Access-Control-Allow-Origin: *");
}
header('Access-Control-Allow-Credentials: true');
header('Access-Control-Max-Age: 86400');
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    exit;
}

header("Content-Type: application/json; charset=UTF-8");

require_once 'vendor/autoload.php';

$dotenv = Dotenv\Dotenv::createImmutable(__DIR__);
$dotenv->load();

require_once 'config/db.php';

// Simple Router
$request_uri = $_SERVER['REQUEST_URI'];
$base_path = '/helafood-api';
$path = str_replace($base_path, '', $request_uri);
$path = explode('?', $path)[0];

if ($path === '/login' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    require 'auth/login.php';
} elseif ($path === '/validate' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    require 'auth/validateToken.php';
} elseif ($path === '/recipes') {
    require 'api/recipes.php';
} elseif (strpos($path, '/interactions') === 0) {
    require 'api/interactions.php';
} elseif (strpos($path, '/categories') === 0) {
    require 'api/categories.php';
} else {
    if (ob_get_length()) ob_end_clean();
    http_response_code(404);
    echo json_encode(["message" => "Endpoint not found", "path" => $path]);
}
